//
//  tirem_epm.c
//  TVStudy
//
//  Copyright (c) 2017 Hammett & Edison, Inc.  All rights reserved.


// Propagation modelling using H&E modefied version of TIREM, see the lib_tirem.f module for details.


#ifdef __BUILD_TVSTUDY
#include "../tvstudy.h"
#else
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#include "../global.h"
#include "model.h"
#include "../fcc_curve.h"
#endif


//---------------------------------------------------------------------------------------------------------------------

#define METERS_TO_FEET       3.280839895
#define KILOMETERS_TO_MILES  0.621369949


//---------------------------------------------------------------------------------------------------------------------

void tirem_(float *h, int *npt, float *psp, float *txh, float *rxh, float *frq, int *ihv, int *iep, float *xns,
	float *sig, float *per, float *fslo, float *plos, int *imod);

//---------------------------------------------------------------------------------------------------------------------
// Compute field strength at a specified location using TIREM supplemented by the EPM-73 model.

// Arguments:

//    data  Model data structure.

// Return 0 on success, non-zero if path loss calculation fails.

int tirem_epm(MODEL_DATA *data) {

	// Profile point count has a hard limit.

	if (data->profileCount > TIREM_EPM_NELMAX) {
		snprintf(data->errorMessage, MESSAGE_LEN, "**TIREM failed: bad profile point count");
		return 1;
	}

	// Convert profile elevations to feet, set up other parameters to pass to FORTRAN routine.  Always put the EPM-73
	// modification in "automatic" mode which selects based on frequency.  

	float elev[TIREM_EPM_NELMAX];
	int i;
	for (i = 0; i < data->profileCount; i++) {
		elev[i] = data->profile[i] * METERS_TO_FEET;
	}
	int nels = data->profileCount;
    float dinc = (float)((1. / data->profilePpk) * KILOMETERS_TO_MILES);
	float tht = (float)((data->profile[0] + data->transmitHeightAGL) * METERS_TO_FEET);
    float hsr = (float)data->receiveHeightAGL * METERS_TO_FEET;
    float freq = (float)data->frequency;
	int ipol = data->signalPolarization;
	int kepm = 2;
    float xns = (float)data->atmosphericRefractivity;
    float sigma = (float)data->groundConductivity;
    float eps = (float)data->groundPermittivity;

	// Run TIREM, check for errors.  A detected error sets model key to 0, but undetected errors can also occur so
	// check the loss value for infinite, NaN, or negative loss relative to free-space.  If so set the error code in
	// the data structure and set the path loss to free-space.

	float fslo = 0., rlos = 0.;
	int imod = 0;
	tirem_(elev, &nels, &dinc, &tht, &hsr, &freq, &ipol, &kepm, &xns, &sigma, &eps, &fslo, &rlos, &imod);
	double pathLoss = (double)rlos;
	if (!imod || (rlos < fslo) || !isfinite(pathLoss)) {
		if (!imod) {
			data->errorCode = 1;
		} else {
			if (rlos < fslo) {
				data->errorCode = 2;
			} else {
				data->errorCode = 3;
			}
		}
		pathLoss = (double)fslo;
	}

	// The TIREM code always returns median pathloss, i.e. F(50,50).  However F(50,10) or F(50,90) may be simulated by
	// using the FCC curves to compute a difference between curve sets and apply to the median loss.  Arbitrarily set
	// the median range between 30 and 70 percent time variability.	 First compute the HAAT.

	if ((data->percentTime < 30.) || (data->percentTime > 70.)) {

		double avet = 0.;
		int i1 = 3.2 * data->profilePpk, i2 = 16.1 * data->profilePpk;
		if (i2 > (data->profileCount - 1)) {
			i2 = data->profileCount - 1;
		}
		for (i = i1; i <= i2; i++) {
			avet += data->profile[i];
		}
		avet /= (double)(i2 - i1 + 1);

		double haat = (data->profile[0] + data->transmitHeightAGL) - avet;
		if (haat < 30.5) {
			haat = 30.5;
		}

		// If the distance is beyond the curve end just use the last point.

		double dist = data->distance;
		if (dist > 300.) {
			dist = 300.;
		}

		int curv = FCC_F90;
		if (data->percentTime < 30.) {
			curv = FCC_F10;
		}

		int band;
		if (data->frequency < 74.) {
			band = BAND_VLO1;
		} else {
			if (data->frequency < 131.) {
				band = BAND_VLO2;
			} else {
				if (data->frequency < 300.) {
					band = BAND_VHI;
				} else {
					band = BAND_UHF;
				}
			}
		}

		// First look up on F(50,50), then on the actual curve and apply delta.  Ignore errors, there shouldn't be any.

		double erp = 0., sig50 = 0., sig = 0.;
#ifdef __BUILD_TVSTUDY
		fcc_curve(&erp, &sig50, &dist, haat, band, FCC_FLD, FCC_F50, OFF_CURV_METH_FS, NULL, 0., NULL, NULL, NULL);
		fcc_curve(&erp, &sig, &dist, haat, band, FCC_FLD, curv, OFF_CURV_METH_FS, NULL, 0., NULL, NULL, NULL);
#else
		fcc_curve(&erp, &sig50, &dist, haat, band, FCC_FLD, FCC_F50, OFF_CURV_METH_FS);
		fcc_curve(&erp, &sig, &dist, haat, band, FCC_FLD, curv, OFF_CURV_METH_FS);
#endif
		pathLoss += sig50 - sig;
	}

	// Compute field strength for 0 dBk.

	data->fieldStrength = (20. * log10(data->frequency)) + 139.3655 - pathLoss;

	return 0;
}
